<?php

use App\Http\Controllers\Api\ApiController;
use App\Http\Controllers\WebhookController;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Route;

/*
|--------------------------------------------------------------------------
| API Routes
|--------------------------------------------------------------------------
|
| Here is where you can register API routes for your application. These
| routes are loaded by the RouteServiceProvider and all of them will
| be assigned to the "api" middleware group. Make something great!
|
*/

// Public API routes
Route::middleware(['throttle:api'])->group(function () {
    // Trust seal display endpoint (public)
    Route::get('/trust-seal/{seal}', [ApiController::class, 'getTrustSeal'])
        ->name('api.trust-seal.show')
        ->middleware('throttle:trust-seal');
    
    // Trust seal view tracking (public)
    Route::post('/trust-seal/{seal}/view', [ApiController::class, 'trackSealView'])
        ->name('api.trust-seal.track')
        ->middleware('throttle:trust-seal');
});

// Webhook routes (no authentication required)
Route::prefix('webhooks')->group(function () {
    Route::post('/stripe', [WebhookController::class, 'handleStripe'])
        ->name('api.webhooks.stripe')
        ->middleware('throttle:webhooks');
    
    Route::post('/security', [WebhookController::class, 'handleSecurity'])
        ->name('api.webhooks.security')
        ->middleware('throttle:webhooks');
    
    Route::post('/analytics', [WebhookController::class, 'handleAnalytics'])
        ->name('api.webhooks.analytics')
        ->middleware('throttle:webhooks');
});

// Authenticated API routes
Route::middleware(['auth:sanctum', 'throttle:api'])->group(function () {
    // User profile
    Route::get('/user', function (Request $request) {
        return $request->user();
    });
    
    // Domain management
    Route::prefix('domains')->group(function () {
        Route::get('/', [ApiController::class, 'getDomains'])
            ->name('api.domains.index');
        
        Route::post('/', [ApiController::class, 'addDomain'])
            ->name('api.domains.store')
            ->middleware('throttle:domain-verification');
        
        Route::get('/{domain}', [ApiController::class, 'getDomainDetails'])
            ->name('api.domains.show')
            ->middleware('verify-domain-ownership');
        
        Route::put('/{domain}/status', [ApiController::class, 'updateDomainStatus'])
            ->name('api.domains.update-status')
            ->middleware('verify-domain-ownership');
        
        Route::delete('/{domain}', [ApiController::class, 'deleteDomain'])
            ->name('api.domains.destroy')
            ->middleware('verify-domain-ownership');
    });
    
    // Analytics
    Route::prefix('analytics')->group(function () {
        Route::get('/', [ApiController::class, 'getAnalytics'])
            ->name('api.analytics.index');
        
        Route::get('/domain/{domain}', [ApiController::class, 'getDomainAnalytics'])
            ->name('api.analytics.domain')
            ->middleware('verify-domain-ownership');
        
        Route::get('/export', [ApiController::class, 'exportAnalytics'])
            ->name('api.analytics.export')
            ->middleware('throttle:exports');
        
        Route::get('/export/domain/{domain}', [ApiController::class, 'exportAnalytics'])
            ->name('api.analytics.export-domain')
            ->middleware(['verify-domain-ownership', 'throttle:exports']);
    });
    
    // Trust seal configuration
    Route::prefix('trust-seals')->group(function () {
        Route::get('/', [ApiController::class, 'getTrustSeals'])
            ->name('api.trust-seals.index');
        
        Route::get('/{seal}', [ApiController::class, 'getTrustSealConfig'])
            ->name('api.trust-seals.show')
            ->middleware('verify-domain-ownership:seal');
        
        Route::put('/{seal}', [ApiController::class, 'updateTrustSealConfig'])
            ->name('api.trust-seals.update')
            ->middleware('verify-domain-ownership:seal');
    });
    
    // Security monitoring
    Route::prefix('security')->group(function () {
        Route::get('/checks', [ApiController::class, 'getSecurityChecks'])
            ->name('api.security.checks');
        
        Route::post('/check/{domain}', [ApiController::class, 'runSecurityCheck'])
            ->name('api.security.run-check')
            ->middleware(['verify-domain-ownership', 'throttle:security-checks']);
        
        Route::get('/alerts', [ApiController::class, 'getSecurityAlerts'])
            ->name('api.security.alerts');
        
        Route::get('/alerts/domain/{domain}', [ApiController::class, 'getSecurityAlerts'])
            ->name('api.security.alerts-domain')
            ->middleware('verify-domain-ownership');
    });
    
    // Statistics and reporting
    Route::prefix('stats')->group(function () {
        Route::get('/', [ApiController::class, 'getStats'])
            ->name('api.stats.index');
        
        Route::get('/real-time', [ApiController::class, 'getRealTimeStats'])
            ->name('api.stats.real-time');
        
        Route::get('/domain/{domain}', [ApiController::class, 'getDomainStats'])
            ->name('api.stats.domain')
            ->middleware('verify-domain-ownership');
    });
    
    // API usage and limits
    Route::get('/usage', [ApiController::class, 'apiUsage'])
        ->name('api.usage');
});

// Premium API routes
Route::middleware(['auth:sanctum', 'throttle:api-premium', 'can:premium'])->group(function () {
    // Advanced analytics
    Route::prefix('premium/analytics')->group(function () {
        Route::get('/advanced', [ApiController::class, 'getAdvancedAnalytics'])
            ->name('api.premium.analytics.advanced');
        
        Route::get('/trends', [ApiController::class, 'getAnalyticsTrends'])
            ->name('api.premium.analytics.trends');
        
        Route::get('/reports', [ApiController::class, 'getAnalyticsReports'])
            ->name('api.premium.analytics.reports');
    });
    
    // Bulk operations
    Route::prefix('premium/bulk')->group(function () {
        Route::post('/security-checks', [ApiController::class, 'bulkSecurityChecks'])
            ->name('api.premium.bulk.security-checks')
            ->middleware('throttle:bulk-operations');
        
        Route::post('/domain-updates', [ApiController::class, 'bulkDomainUpdates'])
            ->name('api.premium.bulk.domain-updates')
            ->middleware('throttle:bulk-operations');
    });
    
    // API token management (premium only)
    Route::prefix('premium/tokens')->group(function () {
        Route::get('/', [ApiController::class, 'getApiTokens'])
            ->name('api.premium.tokens.index');
        
        Route::post('/', [ApiController::class, 'createApiToken'])
            ->name('api.premium.tokens.store');
        
        Route::delete('/{token}', [ApiController::class, 'deleteApiToken'])
            ->name('api.premium.tokens.destroy');
    });
});

// Admin API routes
Route::middleware(['auth:sanctum', 'throttle:api', 'can:admin'])->prefix('admin')->group(function () {
    // System statistics
    Route::get('/system/stats', [ApiController::class, 'getSystemStats'])
        ->name('api.admin.system.stats');
    
    // User management
    Route::prefix('users')->group(function () {
        Route::get('/', [ApiController::class, 'getUsers'])
            ->name('api.admin.users.index');
        
        Route::get('/{user}', [ApiController::class, 'getUserDetails'])
            ->name('api.admin.users.show');
        
        Route::put('/{user}/status', [ApiController::class, 'updateUserStatus'])
            ->name('api.admin.users.update-status');
    });
    
    // Domain management
    Route::prefix('domains')->group(function () {
        Route::get('/', [ApiController::class, 'getAllDomains'])
            ->name('api.admin.domains.index');
        
        Route::put('/{domain}/force-verify', [ApiController::class, 'forceVerifyDomain'])
            ->name('api.admin.domains.force-verify');
        
        Route::delete('/{domain}/force-delete', [ApiController::class, 'forceDeleteDomain'])
            ->name('api.admin.domains.force-delete');
    });
    
    // Security monitoring
    Route::prefix('security')->group(function () {
        Route::get('/global-alerts', [ApiController::class, 'getGlobalSecurityAlerts'])
            ->name('api.admin.security.global-alerts');
        
        Route::post('/run-global-check', [ApiController::class, 'runGlobalSecurityCheck'])
            ->name('api.admin.security.run-global-check');
    });
    
    // System maintenance
    Route::prefix('maintenance')->group(function () {
        Route::post('/clear-cache', [ApiController::class, 'clearCache'])
            ->name('api.admin.maintenance.clear-cache');
        
        Route::post('/process-analytics', [ApiController::class, 'processAnalytics'])
            ->name('api.admin.maintenance.process-analytics');
        
        Route::get('/system-health', [ApiController::class, 'getSystemHealth'])
            ->name('api.admin.maintenance.system-health');
    });
});