<?php

namespace App\Services;

use App\Models\Domain;
use App\Models\DomainVerification;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Carbon\Carbon;

class DomainVerificationService
{
    /**
     * Verify domain using specified method
     */
    public function verifyDomain(Domain $domain, string $method): array
    {
        try {
            // Create or update verification record
            $verification = $this->createVerificationRecord($domain, $method);
            
            $result = match($method) {
                'dns' => $this->verifyDNS($domain, $verification),
                'file' => $this->verifyFile($domain, $verification),
                'meta' => $this->verifyMeta($domain, $verification),
                default => ['success' => false, 'message' => 'Invalid verification method']
            };
            
            // Update verification record with result
            $this->updateVerificationRecord($verification, $result);
            
            // Update domain status if verification successful
            if ($result['success']) {
                $domain->update([
                    'status' => 'verified',
                    'verified_at' => now()
                ]);
            }
            
            return $result;
            
        } catch (\Exception $e) {
            Log::error('Domain verification failed', [
                'domain' => $domain->domain,
                'method' => $method,
                'error' => $e->getMessage()
            ]);
            
            return [
                'success' => false,
                'message' => 'Verification failed: ' . $e->getMessage()
            ];
        }
    }
    
    /**
     * Verify DNS TXT record
     */
    private function verifyDNS(Domain $domain, DomainVerification $verification): array
    {
        $domainName = $domain->domain;
        $expectedValue = $domain->verification_token;
        
        try {
            // Query DNS TXT records for _trustseal subdomain
            $records = dns_get_record("_trustseal.{$domainName}", DNS_TXT);
            
            if (empty($records)) {
                return [
                    'success' => false,
                    'message' => 'No TXT record found for _trustseal subdomain'
                ];
            }
            
            // Check if any record matches our token
            foreach ($records as $record) {
                if (isset($record['txt']) && trim($record['txt']) === $expectedValue) {
                    return [
                        'success' => true,
                        'message' => 'DNS verification successful',
                        'found_value' => $record['txt']
                    ];
                }
            }
            
            return [
                'success' => false,
                'message' => 'TXT record found but token does not match',
                'found_value' => $records[0]['txt'] ?? 'N/A'
            ];
            
        } catch (\Exception $e) {
            return [
                'success' => false,
                'message' => 'DNS lookup failed: ' . $e->getMessage()
            ];
        }
    }
    
    /**
     * Verify HTML file upload
     */
    private function verifyFile(Domain $domain, DomainVerification $verification): array
    {
        $domainName = $domain->domain;
        $expectedValue = $domain->verification_token;
        $fileUrl = "http://{$domainName}/trustseal-verification.txt";
        
        try {
            // Try HTTP first, then HTTPS
            $response = $this->fetchUrl($fileUrl);
            
            if (!$response['success']) {
                $httpsUrl = "https://{$domainName}/trustseal-verification.txt";
                $response = $this->fetchUrl($httpsUrl);
            }
            
            if (!$response['success']) {
                return [
                    'success' => false,
                    'message' => 'Verification file not accessible: ' . $response['error']
                ];
            }
            
            $content = trim($response['content']);
            
            if ($content === $expectedValue) {
                return [
                    'success' => true,
                    'message' => 'File verification successful',
                    'found_value' => $content
                ];
            } else {
                return [
                    'success' => false,
                    'message' => 'File found but content does not match expected token',
                    'found_value' => $content
                ];
            }
            
        } catch (\Exception $e) {
            return [
                'success' => false,
                'message' => 'File verification failed: ' . $e->getMessage()
            ];
        }
    }
    
    /**
     * Verify meta tag
     */
    private function verifyMeta(Domain $domain, DomainVerification $verification): array
    {
        $domainName = $domain->domain;
        $expectedValue = $domain->verification_token;
        
        try {
            // Try HTTP first, then HTTPS
            $homeUrl = "http://{$domainName}";
            $response = $this->fetchUrl($homeUrl);
            
            if (!$response['success']) {
                $httpsUrl = "https://{$domainName}";
                $response = $this->fetchUrl($httpsUrl);
            }
            
            if (!$response['success']) {
                return [
                    'success' => false,
                    'message' => 'Unable to access website homepage: ' . $response['error']
                ];
            }
            
            $html = $response['content'];
            
            // Look for the meta tag
            $pattern = '/<meta\s+name=["\']trustseal-verification["\']\s+content=["\']([^"\'\/]*)["\']\s*\/?>/i';
            
            if (preg_match($pattern, $html, $matches)) {
                $foundValue = $matches[1];
                
                if ($foundValue === $expectedValue) {
                    return [
                        'success' => true,
                        'message' => 'Meta tag verification successful',
                        'found_value' => $foundValue
                    ];
                } else {
                    return [
                        'success' => false,
                        'message' => 'Meta tag found but token does not match',
                        'found_value' => $foundValue
                    ];
                }
            } else {
                return [
                    'success' => false,
                    'message' => 'Meta tag not found in homepage HTML'
                ];
            }
            
        } catch (\Exception $e) {
            return [
                'success' => false,
                'message' => 'Meta tag verification failed: ' . $e->getMessage()
            ];
        }
    }
    
    /**
     * Fetch URL content with error handling
     */
    private function fetchUrl(string $url): array
    {
        try {
            $response = Http::timeout(30)
                ->withOptions([
                    'verify' => false, // Skip SSL verification for flexibility
                    'allow_redirects' => true
                ])
                ->get($url);
                
            if ($response->successful()) {
                return [
                    'success' => true,
                    'content' => $response->body()
                ];
            } else {
                return [
                    'success' => false,
                    'error' => "HTTP {$response->status()}: {$response->reason()}"
                ];
            }
            
        } catch (\Exception $e) {
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }
    
    /**
     * Create verification record
     */
    private function createVerificationRecord(Domain $domain, string $method): DomainVerification
    {
        return DomainVerification::create([
            'domain_id' => $domain->id,
            'method' => $method,
            'token' => $domain->verification_token,
            'expected_value' => $domain->verification_token,
            'status' => 'pending',
            'attempts' => 0,
            'expires_at' => now()->addDays(7)
        ]);
    }
    
    /**
     * Update verification record with result
     */
    private function updateVerificationRecord(DomainVerification $verification, array $result): void
    {
        $verification->update([
            'status' => $result['success'] ? 'verified' : 'failed',
            'found_value' => $result['found_value'] ?? null,
            'error_message' => $result['success'] ? null : $result['message'],
            'verified_at' => $result['success'] ? now() : null,
            'attempts' => $verification->attempts + 1,
            'last_attempt_at' => now()
        ]);
    }
    
    /**
     * Check if domain can retry verification
     */
    public function canRetryVerification(Domain $domain, string $method): bool
    {
        $recentVerification = $domain->domainVerifications()
            ->where('method', $method)
            ->where('last_attempt_at', '>=', now()->subHour())
            ->first();
            
        if (!$recentVerification) {
            return true;
        }
        
        return $recentVerification->attempts < 5;
    }
    
    /**
     * Get verification status for domain
     */
    public function getVerificationStatus(Domain $domain): array
    {
        $verifications = $domain->domainVerifications()
            ->latest('last_attempt_at')
            ->get()
            ->groupBy('method');
            
        $status = [];
        
        foreach (['dns', 'file', 'meta'] as $method) {
            $latestVerification = $verifications->get($method)?->first();
            
            $status[$method] = [
                'attempted' => (bool) $latestVerification,
                'status' => $latestVerification?->status ?? 'not_attempted',
                'last_attempt' => $latestVerification?->last_attempt_at,
                'can_retry' => $this->canRetryVerification($domain, $method),
                'error_message' => $latestVerification?->error_message
            ];
        }
        
        return $status;
    }
}