<?php

namespace App\Providers;

use Illuminate\Cache\RateLimiting\Limit;
use Illuminate\Foundation\Support\Providers\RouteServiceProvider as ServiceProvider;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\RateLimiter;
use Illuminate\Support\Facades\Route;

class RouteServiceProvider extends ServiceProvider
{
    /**
     * The path to your application's "home" route.
     *
     * Typically, users are redirected here after authentication.
     *
     * @var string
     */
    public const HOME = '/dashboard';

    /**
     * Define your route model bindings, pattern filters, and other route configuration.
     */
    public function boot(): void
    {
        $this->configureRateLimiting();
        $this->configureRouteModelBinding();

        $this->routes(function () {
            Route::middleware('api')
                ->prefix('api')
                ->group(base_path('routes/api.php'));

            Route::middleware('web')
                ->group(base_path('routes/web.php'));

            // Admin routes (if they exist)
            if (file_exists(base_path('routes/admin.php'))) {
                Route::middleware(['web', 'auth', 'admin'])
                    ->prefix('admin')
                    ->name('admin.')
                    ->group(base_path('routes/admin.php'));
            }

            // Console routes (if they exist)
            if (file_exists(base_path('routes/console.php'))) {
                require base_path('routes/console.php');
            }

            // Channels routes for broadcasting
            if (file_exists(base_path('routes/channels.php'))) {
                require base_path('routes/channels.php');
            }
        });
    }

    /**
     * Configure the rate limiters for the application.
     */
    protected function configureRateLimiting(): void
    {
        // General API rate limiting
        RateLimiter::for('api', function (Request $request) {
            $user = $request->user();
            
            if ($user) {
                // Premium users get higher limits
                $limit = $user->isPremium() ? 300 : 60;
                return Limit::perMinute($limit)->by($user->id);
            }
            
            // Non-authenticated requests limited by IP
            return Limit::perMinute(30)->by($request->ip());
        });

        // Strict rate limiting for authentication endpoints
        RateLimiter::for('auth', function (Request $request) {
            return Limit::perMinute(5)->by($request->ip());
        });

        // Rate limiting for trust seal display (public endpoint)
        RateLimiter::for('seal-display', function (Request $request) {
            return Limit::perMinute(1000)->by($request->ip());
        });

        // Domain verification rate limiting
        RateLimiter::for('domain-verification', function (Request $request) {
            $user = $request->user();
            if ($user) {
                return Limit::perHour(10)->by($user->id);
            }
            return Limit::perHour(5)->by($request->ip());
        });

        // Security check rate limiting
        RateLimiter::for('security-check', function (Request $request) {
            $user = $request->user();
            if ($user) {
                $limit = $user->isPremium() ? 50 : 20;
                return Limit::perHour($limit)->by($user->id);
            }
            return Limit::perHour(10)->by($request->ip());
        });

        // Contact form rate limiting
        RateLimiter::for('contact', function (Request $request) {
            return Limit::perHour(3)->by($request->ip());
        });

        // Password reset rate limiting
        RateLimiter::for('password-reset', function (Request $request) {
            return Limit::perHour(5)->by($request->ip());
        });

        // Registration rate limiting
        RateLimiter::for('register', function (Request $request) {
            return Limit::perHour(10)->by($request->ip());
        });

        // Email verification rate limiting
        RateLimiter::for('email-verification', function (Request $request) {
            return Limit::perHour(6)->by($request->ip());
        });

        // Webhook rate limiting
        RateLimiter::for('webhook', function (Request $request) {
            return Limit::perMinute(100)->by($request->ip());
        });

        // Export rate limiting
        RateLimiter::for('export', function (Request $request) {
            $user = $request->user();
            if ($user) {
                return Limit::perHour(5)->by($user->id);
            }
            return Limit::perHour(2)->by($request->ip());
        });

        // Global rate limiting (fallback)
        RateLimiter::for('global', function (Request $request) {
            return Limit::perMinute(1000)->by($request->ip());
        });
    }

    /**
     * Configure route model binding.
     */
    protected function configureRouteModelBinding(): void
    {
        // Domain binding - ensure user owns the domain
        Route::bind('domain', function (string $value) {
            $domain = \App\Models\Domain::where('id', $value)
                ->orWhere('domain', $value)
                ->first();

            if (!$domain) {
                abort(404, 'Domain not found');
            }

            // Check ownership if user is authenticated
            if (auth()->check() && $domain->user_id !== auth()->user()->id) {
                abort(403, 'Access denied');
            }

            return $domain;
        });

        // Trust Seal binding - ensure user owns the seal's domain
        Route::bind('trustSeal', function (string $value) {
            $trustSeal = \App\Models\TrustSeal::with('domain')->find($value);

            if (!$trustSeal) {
                abort(404, 'Trust seal not found');
            }

            // Check ownership if user is authenticated
            if (auth()->check() && $trustSeal->domain->user_id !== auth()->user()->id) {
                abort(403, 'Access denied');
            }

            return $trustSeal;
        });

        // Subscription binding - ensure user owns the subscription
        Route::bind('subscription', function (string $value) {
            $subscription = \App\Models\Subscription::find($value);

            if (!$subscription) {
                abort(404, 'Subscription not found');
            }

            // Check ownership if user is authenticated
            if (auth()->check() && $subscription->user_id !== auth()->user()->id) {
                abort(403, 'Access denied');
            }

            return $subscription;
        });

        // Security Check binding - ensure user owns the check's domain
        Route::bind('securityCheck', function (string $value) {
            $securityCheck = \App\Models\SecurityCheck::with('domain')->find($value);

            if (!$securityCheck) {
                abort(404, 'Security check not found');
            }

            // Check ownership if user is authenticated
            if (auth()->check() && $securityCheck->domain->user_id !== auth()->user()->id) {
                abort(403, 'Access denied');
            }

            return $securityCheck;
        });

        // API Token binding - ensure user owns the token
        Route::bind('apiToken', function (string $value) {
            $token = \Laravel\Sanctum\PersonalAccessToken::find($value);

            if (!$token) {
                abort(404, 'API token not found');
            }

            // Check ownership if user is authenticated
            if (auth()->check() && $token->tokenable_id !== auth()->user()->id) {
                abort(403, 'Access denied');
            }

            return $token;
        });
    }

    /**
     * Get the route middleware groups.
     */
    public function getMiddlewareGroups(): array
    {
        return [
            'web' => [
                \App\Http\Middleware\EncryptCookies::class,
                \Illuminate\Cookie\Middleware\AddQueuedCookiesToResponse::class,
                \Illuminate\Session\Middleware\StartSession::class,
                // \Illuminate\Session\Middleware\AuthenticateSession::class,
                \Illuminate\View\Middleware\ShareErrorsFromSession::class,
                \App\Http\Middleware\VerifyCsrfToken::class,
                \Illuminate\Routing\Middleware\SubstituteBindings::class,
            ],

            'api' => [
                // \Laravel\Sanctum\Http\Middleware\EnsureFrontendRequestsAreStateful::class,
                'throttle:api',
                \Illuminate\Routing\Middleware\SubstituteBindings::class,
                \App\Http\Middleware\HandleCors::class,
            ],
        ];
    }

    /**
     * Get the route middleware aliases.
     */
    public function getMiddlewareAliases(): array
    {
        return [
            'auth' => \App\Http\Middleware\Authenticate::class,
            'auth.basic' => \Illuminate\Auth\Middleware\AuthenticateWithBasicAuth::class,
            'cache.headers' => \Illuminate\Http\Middleware\SetCacheHeaders::class,
            'can' => \Illuminate\Auth\Middleware\Authorize::class,
            'guest' => \App\Http\Middleware\RedirectIfAuthenticated::class,
            'password.confirm' => \Illuminate\Auth\Middleware\RequirePassword::class,
            'signed' => \Illuminate\Routing\Middleware\ValidateSignature::class,
            'throttle' => \Illuminate\Routing\Middleware\ThrottleRequests::class,
            'verified' => \Illuminate\Auth\Middleware\EnsureEmailIsVerified::class,
            'api.rate.limit' => \App\Http\Middleware\ApiRateLimit::class,
            'domain.owner' => \App\Http\Middleware\VerifyDomainOwnership::class,
            'cors' => \App\Http\Middleware\HandleCors::class,
        ];
    }

    /**
     * Configure additional route patterns.
     */
    protected function configureRoutePatterns(): void
    {
        Route::pattern('id', '[0-9]+');
        Route::pattern('domain', '[a-zA-Z0-9\-\.]+');
        Route::pattern('hash', '[a-zA-Z0-9]+');
        Route::pattern('token', '[a-zA-Z0-9\-_]+');
        Route::pattern('uuid', '[0-9a-f]{8}-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{12}');
    }

    /**
     * Define custom route macros.
     */
    protected function defineRouteMacros(): void
    {
        // Macro for API resource routes with rate limiting
        Route::macro('apiResource', function ($name, $controller, $options = []) {
            $rateLimiter = $options['rate_limiter'] ?? 'api';
            
            return Route::middleware(['throttle:' . $rateLimiter])
                ->resource($name, $controller, $options);
        });

        // Macro for domain-scoped routes
        Route::macro('domainScoped', function ($prefix, $callback) {
            return Route::prefix($prefix)
                ->middleware(['domain.owner'])
                ->group($callback);
        });

        // Macro for authenticated API routes
        Route::macro('authApi', function ($callback) {
            return Route::middleware(['auth:sanctum', 'throttle:api'])
                ->group($callback);
        });
    }
}