<?php

namespace App\Providers;

use App\Events\DomainVerified;
use App\Events\SecurityAlert;
use App\Events\SealViewed;
use App\Listeners\HandleDomainVerification;
use App\Listeners\HandleSecurityAlert;
use App\Listeners\TrackSealView;
use Illuminate\Auth\Events\Registered;
use Illuminate\Auth\Listeners\SendEmailVerificationNotification;
use Illuminate\Foundation\Support\Providers\EventServiceProvider as ServiceProvider;
use Illuminate\Support\Facades\Event;

class EventServiceProvider extends ServiceProvider
{
    /**
     * The event to listener mappings for the application.
     *
     * @var array<class-string, array<int, class-string>>
     */
    protected $listen = [
        Registered::class => [
            SendEmailVerificationNotification::class,
        ],

        // Domain Events
        DomainVerified::class => [
            HandleDomainVerification::class,
        ],

        // Security Events
        SecurityAlert::class => [
            HandleSecurityAlert::class,
        ],

        // Analytics Events
        SealViewed::class => [
            TrackSealView::class,
        ],
    ];

    /**
     * Register any events for your application.
     */
    public function boot(): void
    {
        parent::boot();

        // Additional event listeners can be registered here
        $this->registerModelEvents();
        $this->registerQueueEvents();
        $this->registerAuthEvents();
    }

    /**
     * Determine if events and listeners should be automatically discovered.
     */
    public function shouldDiscoverEvents(): bool
    {
        return false;
    }

    /**
     * Register model events.
     */
    protected function registerModelEvents(): void
    {
        // User model events
        Event::listen('eloquent.created: App\Models\User', function ($user) {
            \Log::info('New user registered', ['user_id' => $user->id, 'email' => $user->email]);
        });

        // Domain model events
        Event::listen('eloquent.created: App\Models\Domain', function ($domain) {
            \Log::info('New domain added', [
                'domain_id' => $domain->id,
                'domain' => $domain->domain,
                'user_id' => $domain->user_id
            ]);
        });

        Event::listen('eloquent.updated: App\Models\Domain', function ($domain) {
            if ($domain->wasChanged('status') && $domain->status === 'verified') {
                // Domain was just verified
                DomainVerified::dispatch(
                    $domain,
                    $domain->verification_method ?? 'unknown',
                    now()
                );
            }
        });

        // Trust Seal model events
        Event::listen('eloquent.created: App\Models\TrustSeal', function ($trustSeal) {
            \Log::info('Trust seal created', [
                'seal_id' => $trustSeal->id,
                'domain_id' => $trustSeal->domain_id,
                'type' => $trustSeal->type
            ]);
        });

        // Subscription model events
        Event::listen('eloquent.created: App\Models\Subscription', function ($subscription) {
            \Log::info('Subscription created', [
                'subscription_id' => $subscription->id,
                'user_id' => $subscription->user_id,
                'plan' => $subscription->plan
            ]);
        });

        Event::listen('eloquent.updated: App\Models\Subscription', function ($subscription) {
            if ($subscription->wasChanged('status')) {
                \Log::info('Subscription status changed', [
                    'subscription_id' => $subscription->id,
                    'old_status' => $subscription->getOriginal('status'),
                    'new_status' => $subscription->status
                ]);
            }
        });

        // Security Check model events
        Event::listen('eloquent.created: App\Models\SecurityCheck', function ($securityCheck) {
            // Dispatch security alert if check failed
            if ($securityCheck->status === 'failed' || $securityCheck->status === 'error') {
                SecurityAlert::dispatch(
                    $securityCheck->domain,
                    $securityCheck->type,
                    $securityCheck->status === 'error' ? 'error' : 'warning',
                    $securityCheck->details['message'] ?? 'Security check failed',
                    $securityCheck->details ?? [],
                    now()
                );
            }
        });
    }

    /**
     * Register queue events.
     */
    protected function registerQueueEvents(): void
    {
        Event::listen('queue.failed', function ($event) {
            \Log::error('Queue job failed', [
                'job' => $event->job->resolveName(),
                'exception' => $event->exception->getMessage(),
                'data' => $event->data
            ]);
        });

        Event::listen('queue.after', function ($event) {
            if ($event->job->resolveName() === 'App\\Jobs\\ProcessSecurityCheck') {
                \Log::info('Security check job completed', [
                    'job_id' => $event->job->getJobId()
                ]);
            }
        });
    }

    /**
     * Register authentication events.
     */
    protected function registerAuthEvents(): void
    {
        Event::listen('auth.login', function ($event) {
            \Log::info('User logged in', [
                'user_id' => $event->user->id,
                'email' => $event->user->email,
                'ip' => request()->ip(),
                'user_agent' => request()->userAgent()
            ]);
        });

        Event::listen('auth.logout', function ($event) {
            \Log::info('User logged out', [
                'user_id' => $event->user->id,
                'email' => $event->user->email
            ]);
        });

        Event::listen('auth.failed', function ($event) {
            \Log::warning('Authentication failed', [
                'email' => $event->credentials['email'] ?? 'unknown',
                'ip' => request()->ip(),
                'user_agent' => request()->userAgent()
            ]);
        });

        Event::listen('password.reset', function ($event) {
            \Log::info('Password reset', [
                'user_id' => $event->user->id,
                'email' => $event->user->email
            ]);
        });
    }

    /**
     * Get the events and handlers.
     */
    public function listens(): array
    {
        return $this->listen;
    }

    /**
     * Register custom event listeners for third-party packages.
     */
    protected function registerPackageEvents(): void
    {
        // Laravel Telescope events
        if (class_exists('\Laravel\Telescope\TelescopeServiceProvider')) {
            Event::listen('telescope.recording', function ($event) {
                \Log::debug('Telescope recording', ['type' => $event->type]);
            });
        }

        // Laravel Sanctum events
        if (class_exists('\Laravel\Sanctum\SanctumServiceProvider')) {
            Event::listen('sanctum.token.created', function ($event) {
                \Log::info('API token created', [
                    'token_id' => $event->token->id,
                    'user_id' => $event->token->tokenable_id
                ]);
            });
        }
    }

    /**
     * Register wildcard event listeners.
     */
    protected function registerWildcardEvents(): void
    {
        // Listen to all model events for debugging (only in local environment)
        if (app()->environment('local')) {
            Event::listen('eloquent.*', function ($eventName, $data) {
                if (config('app.debug_eloquent_events', false)) {
                    \Log::debug('Eloquent event', [
                        'event' => $eventName,
                        'model' => get_class($data[0] ?? null)
                    ]);
                }
            });
        }

        // Listen to all cache events
        Event::listen('cache.*', function ($eventName, $data) {
            if (config('app.debug_cache_events', false)) {
                \Log::debug('Cache event', [
                    'event' => $eventName,
                    'key' => $data[0] ?? null
                ]);
            }
        });
    }
}