<?php

namespace App\Providers;

use App\Models\User;
use App\Services\AnalyticsService;
use App\Services\DomainVerificationService;
use App\Services\SecurityMonitoringService;
use Illuminate\Support\ServiceProvider;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\View;
use Illuminate\Support\Facades\Blade;
use Illuminate\Http\Request;
use Illuminate\Pagination\Paginator;
use Laravel\Sanctum\Sanctum;

class AppServiceProvider extends ServiceProvider
{
    /**
     * Register any application services.
     */
    public function register(): void
    {
        // Register custom services
        $this->registerCustomServices();
        
        // Register third-party services
        // $this->registerThirdPartyServices();
        
        // Register development services
        $this->registerDevelopmentServices();
    }

    /**
     * Bootstrap any application services.
     */
    public function boot(): void
    {
        // Configure database
        $this->configureDatabaseSettings();
        
        // Configure views and blade
        $this->configureViewsAndBlade();
        
        // Configure pagination
        $this->configurePagination();
        
        // Configure gates and policies
        $this->configureAuthorization();
        
        // Configure model observers
        $this->registerModelObservers();
        
        // Configure URL generation
        $this->configureUrls();
        
        // Configure request macros
        $this->configureRequestMacros();
        
        // Configure collection macros
        $this->configureCollectionMacros();
    }

    /**
     * Register custom application services.
     */
    protected function registerCustomServices(): void
    {
        // Analytics Service
        $this->app->singleton(AnalyticsService::class, function ($app) {
            return new AnalyticsService();
        });

        // Domain Verification Service
        $this->app->singleton(DomainVerificationService::class, function ($app) {
            return new DomainVerificationService();
        });

        // Security Monitoring Service
        $this->app->singleton(SecurityMonitoringService::class, function ($app) {
            return new SecurityMonitoringService();
        });

        // Bind interfaces to implementations
        $this->app->bind(
            'App\\Contracts\\AnalyticsServiceInterface',
            'App\\Services\\AnalyticsService'
        );

        $this->app->bind(
            'App\\Contracts\\DomainVerificationServiceInterface',
            'App\\Services\\DomainVerificationService'
        );

        $this->app->bind(
            'App\\Contracts\\SecurityMonitoringServiceInterface',
            'App\\Services\\SecurityMonitoringService'
        );
    }

    /**
     * Register third-party services.
     */
    protected function registerThirdPartyServices(): void
    {
        // Configure Sanctum
        if (class_exists('Laravel\Sanctum\Sanctum')) {
            \Laravel\Sanctum\Sanctum::ignoreMigrations();
        }

        // Register Telescope in non-production environments
        if ($this->app->environment('local', 'testing') && class_exists('Laravel\\Telescope\\TelescopeServiceProvider')) {
            $this->app->register('Laravel\\Telescope\\TelescopeServiceProvider');
            // $this->app->register('App\\Providers\\TelescopeServiceProvider');
        }

        // Register Horizon in non-production environments
        if ($this->app->environment('local', 'staging', 'production') && class_exists('Laravel\\Horizon\\HorizonServiceProvider')) {
            $this->app->register('Laravel\\Horizon\\HorizonServiceProvider');
        }
    }

    /**
     * Register development services.
     */
    protected function registerDevelopmentServices(): void
    {
        if ($this->app->environment('local')) {
            // Register IDE Helper in development
            if (class_exists('Barryvdh\\LaravelIdeHelper\\IdeHelperServiceProvider')) {
                $this->app->register('Barryvdh\\LaravelIdeHelper\\IdeHelperServiceProvider');
            }

            // Register Debug Bar in development
            if (class_exists('Barryvdh\\Debugbar\\ServiceProvider')) {
                $this->app->register('Barryvdh\\Debugbar\\ServiceProvider');
            }
        }
    }

    /**
     * Configure database settings.
     */
    protected function configureDatabaseSettings(): void
    {
        // Set default string length for older MySQL versions
        Schema::defaultStringLength(191);
        
        // Configure database query logging in development
        if ($this->app->environment('local') && config('app.debug')) {
            \DB::listen(function ($query) {
                if ($query->time > 1000) { // Log slow queries (>1 second)
                    \Log::warning('Slow query detected', [
                        'sql' => $query->sql,
                        'bindings' => $query->bindings,
                        'time' => $query->time . 'ms'
                    ]);
                }
            });
        }
    }

    /**
     * Configure views and Blade directives.
     */
    protected function configureViewsAndBlade(): void
    {
        // Share common data with all views
        View::composer('*', function ($view) {
            $view->with([
                'appName' => config('app.name'),
                'appVersion' => config('app.version', '1.0.0'),
                'currentYear' => date('Y'),
            ]);
        });

        // Custom Blade directives
        Blade::directive('currency', function ($expression) {
            return "<?php echo '$' . number_format($expression, 2); ?>";
        });

        Blade::directive('percentage', function ($expression) {
            return "<?php echo number_format($expression, 1) . '%'; ?>";
        });

        Blade::directive('datetime', function ($expression) {
            return "<?php echo ($expression)->format('M j, Y g:i A'); ?>";
        });

        Blade::directive('date', function ($expression) {
            return "<?php echo ($expression)->format('M j, Y'); ?>";
        });

        Blade::directive('timeago', function ($expression) {
            return "<?php echo ($expression)->diffForHumans(); ?>";
        });

        Blade::directive('premium', function () {
            return "<?php if(auth()->check() && auth()->user()->isPremium()): ?>";
        });

        Blade::directive('endpremium', function () {
            return "<?php endif; ?>";
        });

        Blade::directive('admin', function () {
            return "<?php if(auth()->check() && auth()->user()->isAdmin()): ?>";
        });

        Blade::directive('endadmin', function () {
            return "<?php endif; ?>";
        });
    }

    /**
     * Configure pagination.
     */
    protected function configurePagination(): void
    {
        // Use Bootstrap 5 for pagination views
        Paginator::defaultView('pagination::bootstrap-4');
        Paginator::defaultSimpleView('pagination::simple-bootstrap-4');
    }

    /**
     * Configure authorization gates and policies.
     */
    protected function configureAuthorization(): void
    {
        // Admin gate
        Gate::define('admin', function (User $user) {
            return $user->role === 'admin';
        });

        // Premium user gate
        Gate::define('premium', function (User $user) {
            return $user->isPremium();
        });

        // Domain management gates
        Gate::define('manage-domain', function (User $user, $domain) {
            return $user->id === $domain->user_id;
        });

        Gate::define('view-domain', function (User $user, $domain) {
            return $user->id === $domain->user_id;
        });

        // Trust seal management gates
        Gate::define('manage-trust-seal', function (User $user, $trustSeal) {
            return $user->id === $trustSeal->domain->user_id;
        });

        // Analytics gates
        Gate::define('view-analytics', function (User $user, $domain = null) {
            if ($domain) {
                return $user->id === $domain->user_id;
            }
            return true; // User can view their own analytics
        });

        // Subscription gates
        Gate::define('manage-subscription', function (User $user, $subscription = null) {
            if ($subscription) {
                return $user->id === $subscription->user_id;
            }
            return true; // User can manage their own subscription
        });

        // API token gates
        Gate::define('create-api-token', function (User $user) {
            return $user->isPremium();
        });

        // Export data gate
        Gate::define('export-data', function (User $user) {
            return true; // All users can export their data (GDPR)
        });
    }

    /**
     * Register model observers.
     */
    protected function registerModelObservers(): void
    {
        // Register observers if they exist
        $observers = [
            'App\\Models\\User' => 'App\\Observers\\UserObserver',
            'App\\Models\\Domain' => 'App\\Observers\\DomainObserver',
            'App\\Models\\TrustSeal' => 'App\\Observers\\TrustSealObserver',
            'App\\Models\\Subscription' => 'App\\Observers\\SubscriptionObserver',
            'App\\Models\\SecurityCheck' => 'App\\Observers\\SecurityCheckObserver',
        ];

        foreach ($observers as $model => $observer) {
            if (class_exists($model) && class_exists($observer)) {
                $model::observe($observer);
            }
        }
    }

    /**
     * Configure URL generation.
     */
    protected function configureUrls(): void
    {
        // Force HTTPS in production
        if ($this->app->environment('production')) {
            \URL::forceScheme('https');
        }

        // Set trusted proxies if behind a load balancer
        if (config('app.trusted_proxies')) {
            Request::setTrustedProxies(
                config('app.trusted_proxies'),
                Request::HEADER_X_FORWARDED_ALL
            );
        }
    }

    /**
     * Configure request macros.
     */
    protected function configureRequestMacros(): void
    {
        // Check if request is from a bot
        Request::macro('isBot', function () {
            $userAgent = $this->userAgent();
            $bots = [
                'bot', 'crawler', 'spider', 'scraper', 'slurp',
                'googlebot', 'bingbot', 'yandex', 'baidu',
                'facebook', 'twitter', 'linkedin'
            ];
            
            foreach ($bots as $bot) {
                if (stripos($userAgent, $bot) !== false) {
                    return true;
                }
            }
            
            return false;
        });

        // Get client's real IP
        Request::macro('realIp', function () {
            $headers = [
                'HTTP_CF_CONNECTING_IP',
                'HTTP_X_FORWARDED_FOR',
                'HTTP_X_FORWARDED',
                'HTTP_X_CLUSTER_CLIENT_IP',
                'HTTP_FORWARDED_FOR',
                'HTTP_FORWARDED',
                'REMOTE_ADDR'
            ];

            foreach ($headers as $header) {
                if (isset($_SERVER[$header]) && !empty($_SERVER[$header])) {
                    $ips = explode(',', $_SERVER[$header]);
                    $ip = trim($ips[0]);
                    
                    if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE)) {
                        return $ip;
                    }
                }
            }

            return $this->ip();
        });

        // Check if request is AJAX
        Request::macro('isAjax', function () {
            return $this->ajax() || $this->pjax() || $this->expectsJson();
        });
    }

    /**
     * Configure collection macros.
     */
    protected function configureCollectionMacros(): void
    {
        // Collection macro to convert to CSV
        \Illuminate\Support\Collection::macro('toCsv', function ($filename = 'export.csv', $headers = []) {
            $handle = fopen('php://temp', 'r+');
            
            if (!empty($headers)) {
                fputcsv($handle, $headers);
            } elseif ($this->isNotEmpty()) {
                $first = $this->first();
                if (is_array($first)) {
                    fputcsv($handle, array_keys($first));
                } elseif (is_object($first)) {
                    fputcsv($handle, array_keys($first->toArray()));
                }
            }
            
            $this->each(function ($item) use ($handle) {
                if (is_array($item)) {
                    fputcsv($handle, $item);
                } elseif (is_object($item)) {
                    fputcsv($handle, $item->toArray());
                } else {
                    fputcsv($handle, [$item]);
                }
            });
            
            rewind($handle);
            $csv = stream_get_contents($handle);
            fclose($handle);
            
            return response($csv)
                ->header('Content-Type', 'text/csv')
                ->header('Content-Disposition', 'attachment; filename="' . $filename . '"');
        });
        
        // Collection macro for pagination
        \Illuminate\Support\Collection::macro('paginate', function ($perPage = 15, $page = null, $options = []) {
            $page = $page ?: (\Illuminate\Pagination\Paginator::resolveCurrentPage() ?: 1);
            
            return new \Illuminate\Pagination\LengthAwarePaginator(
                $this->forPage($page, $perPage)->values(),
                $this->count(),
                $perPage,
                $page,
                array_merge($options, [
                    'path' => \Illuminate\Pagination\Paginator::resolveCurrentPath()
                ])
            );
        });
    }
}
