<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Support\Str;

class TrustSeal extends Model
{
    use HasFactory;

    protected $fillable = [
        'domain_id',
        'seal_type',
        'seal_code',
        'customization',
        'is_active',
        'display_count',
        'first_displayed_at',
        'last_displayed_at'
    ];

    protected $casts = [
        'customization' => 'array',
        'is_active' => 'boolean',
        'first_displayed_at' => 'datetime',
        'last_displayed_at' => 'datetime',
    ];

    protected static function boot()
    {
        parent::boot();
        
        static::creating(function ($model) {
            if (empty($model->seal_code)) {
                $model->seal_code = Str::random(32);
            }
        });
    }

    public function domain(): BelongsTo
    {
        return $this->belongsTo(Domain::class);
    }

    public function analytics(): HasMany
    {
        return $this->hasMany(SealAnalytics::class);
    }

    public function getEmbedCodeAttribute(): string
    {
        $url = url('/seal/' . $this->seal_code);
        $width = $this->customization['width'] ?? 120;
        $height = $this->customization['height'] ?? 60;
        
        return '<iframe src="' . $url . '" width="' . $width . '" height="' . $height . '" frameborder="0" scrolling="no" title="Trust Seal"></iframe>';
    }

    public function getDirectLinkAttribute(): string
    {
        return url('/seal/' . $this->seal_code);
    }

    public function incrementDisplayCount(): void
    {
        $this->increment('display_count');
        
        if (is_null($this->first_displayed_at)) {
            $this->update(['first_displayed_at' => now()]);
        }
        
        $this->update(['last_displayed_at' => now()]);
    }

    public function getDefaultCustomization(): array
    {
        return [
            'width' => 120,
            'height' => 60,
            'background_color' => '#ffffff',
            'text_color' => '#333333',
            'border_color' => '#cccccc',
            'accent_color' => '#007bff',
            'show_score' => true,
            'show_timestamp' => true,
            'show_domain' => true,
            'custom_text' => null,
            'logo_url' => null,
            'theme' => 'light' // light, dark, minimal
        ];
    }

    public function getCustomizationAttribute($value): array
    {
        $default = $this->getDefaultCustomization();
        $custom = $value ? json_decode($value, true) : [];
        
        return array_merge($default, $custom);
    }

    public function getSealTypeDisplayAttribute(): string
    {
        return $this->seal_type === 'security' ? 'Security Seal' : 'Trust Seal';
    }

    public function getSealBadgeColorAttribute(): string
    {
        return $this->seal_type === 'security' ? 'success' : 'primary';
    }

    public function canCustomize(): bool
    {
        return $this->domain->user->is_premium;
    }

    public function generateSealHtml(): string
    {
        $customization = $this->customization;
        $domain = $this->domain;
        $securityScore = $domain->security_score;
        $verifiedDate = $domain->verified_at->format('M d, Y');
        
        $html = '<div class="trust-seal" style="';
        $html .= 'width: ' . $customization['width'] . 'px; ';
        $html .= 'height: ' . $customization['height'] . 'px; ';
        $html .= 'background: ' . $customization['background_color'] . '; ';
        $html .= 'color: ' . $customization['text_color'] . '; ';
        $html .= 'border: 1px solid ' . $customization['border_color'] . '; ';
        $html .= 'border-radius: 4px; padding: 8px; font-family: Arial, sans-serif; font-size: 10px; text-align: center;';
        $html .= '">';
        
        if ($customization['show_domain']) {
            $html .= '<div style="font-weight: bold; margin-bottom: 2px;">' . $domain->clean_domain . '</div>';
        }
        
        $html .= '<div style="color: ' . $customization['accent_color'] . '; font-weight: bold;">';
        $html .= $this->seal_type_display;
        $html .= '</div>';
        
        if ($customization['show_score'] && $this->seal_type === 'security') {
            $html .= '<div style="margin: 2px 0;">Score: ' . $securityScore . '/100</div>';
        }
        
        if ($customization['show_timestamp']) {
            $html .= '<div style="font-size: 8px; color: #666;">Verified: ' . $verifiedDate . '</div>';
        }
        
        if ($customization['custom_text']) {
            $html .= '<div style="font-size: 8px; margin-top: 2px;">' . htmlspecialchars($customization['custom_text']) . '</div>';
        }
        
        $html .= '</div>';
        
        return $html;
    }
}