<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class DomainVerification extends Model
{
    use HasFactory;

    protected $fillable = [
        'domain_id',
        'method',
        'token',
        'expected_value',
        'found_value',
        'status',
        'error_message',
        'verified_at',
        'expires_at',
        'attempts',
        'last_attempt_at'
    ];

    protected $casts = [
        'verified_at' => 'datetime',
        'expires_at' => 'datetime',
        'last_attempt_at' => 'datetime',
    ];

    public function domain(): BelongsTo
    {
        return $this->belongsTo(Domain::class);
    }

    public function isExpired(): bool
    {
        return $this->expires_at->isPast();
    }

    public function isVerified(): bool
    {
        return $this->status === 'verified';
    }

    public function canRetry(): bool
    {
        return $this->attempts < 5 && !$this->isExpired();
    }

    public function getVerificationInstructions(): array
    {
        $domain = $this->domain->clean_domain;
        
        switch ($this->method) {
            case 'dns':
                return [
                    'title' => 'DNS TXT Record Verification',
                    'description' => 'Add a TXT record to your domain\'s DNS settings.',
                    'steps' => [
                        'Log in to your domain registrar or DNS provider',
                        'Navigate to DNS management for ' . $domain,
                        'Add a new TXT record with the following details:',
                        'Host/Name: _trustseal',
                        'Value: ' . $this->expected_value,
                        'TTL: 300 (or default)',
                        'Save the record and wait for DNS propagation (up to 24 hours)'
                    ]
                ];
                
            case 'file':
                return [
                    'title' => 'HTML File Upload Verification',
                    'description' => 'Upload a verification file to your website root directory.',
                    'steps' => [
                        'Download the verification file: trust-seal-verification.html',
                        'Upload it to your website\'s root directory (same folder as index.html)',
                        'Make sure the file is accessible at: https://' . $domain . '/trust-seal-verification.html',
                        'The file should contain: ' . $this->expected_value,
                        'Click verify once the file is uploaded'
                    ]
                ];
                
            case 'meta':
                return [
                    'title' => 'HTML Meta Tag Verification',
                    'description' => 'Add a meta tag to your website\'s homepage.',
                    'steps' => [
                        'Open your website\'s homepage HTML file',
                        'Add the following meta tag in the <head> section:',
                        '<meta name="trust-seal-verification" content="' . $this->expected_value . '">',
                        'Save and upload the file to your web server',
                        'Click verify once the meta tag is added'
                    ]
                ];
                
            default:
                return ['title' => 'Unknown verification method'];
        }
    }
}