<?php

namespace App\Listeners;

use App\Events\SecurityAlert;
use App\Models\SecurityAlert as SecurityAlertModel;
use App\Notifications\SecurityAlertNotification;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Cache;

class HandleSecurityAlert implements ShouldQueue
{
    use InteractsWithQueue;

    /**
     * Handle the event.
     */
    public function handle(SecurityAlert $event): void
    {
        $domain = $event->domain;
        
        try {
            // 1. Store the security alert
            $alertModel = $this->storeSecurityAlert($event);
            
            // 2. Send notification based on alert level and user preferences
            $this->sendNotificationIfNeeded($event, $alertModel);
            
            // 3. Update domain security status if needed
            $this->updateDomainSecurityStatus($event);
            
            // 4. Log the alert
            $this->logSecurityAlert($event);
            
            // 5. Update cache and statistics
            $this->updateSecurityStats($event);
            
        } catch (\Exception $e) {
            Log::error('Failed to handle security alert', [
                'domain_id' => $domain->id,
                'domain' => $domain->domain,
                'alert_type' => $event->alertType,
                'alert_level' => $event->alertLevel,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            // Re-throw to trigger retry
            throw $e;
        }
    }
    
    /**
     * Store the security alert in database.
     */
    protected function storeSecurityAlert(SecurityAlert $event): SecurityAlertModel
    {
        return SecurityAlertModel::create([
            'domain_id' => $event->domain->id,
            'alert_type' => $event->alertType,
            'alert_level' => $event->alertLevel,
            'message' => $event->message,
            'details' => $event->details,
            'is_resolved' => false,
            'resolved_at' => null,
            'created_at' => $event->alertedAt,
            'updated_at' => $event->alertedAt
        ]);
    }
    
    /**
     * Send notification if needed based on user preferences.
     */
    protected function sendNotificationIfNeeded(SecurityAlert $event, SecurityAlertModel $alertModel): void
    {
        $domain = $event->domain;
        $user = $domain->user;
        
        // Check user notification preferences
        $preferences = $user->preferences ?? [];
        $notifications = $preferences['notifications'] ?? [];
        
        // Check if user wants security alerts
        if (!($notifications['security_alerts'] ?? true)) {
            return;
        }
        
        // Check if this alert level should trigger notification
        $minLevel = $notifications['security_alert_level'] ?? 'warning';
        if (!$this->shouldNotifyForLevel($event->alertLevel, $minLevel)) {
            return;
        }
        
        // Check rate limiting - don't spam user with same alert type
        if ($this->isRateLimited($domain, $event->alertType)) {
            Log::info('Security alert notification rate limited', [
                'domain_id' => $domain->id,
                'alert_type' => $event->alertType,
                'user_id' => $user->id
            ]);
            return;
        }
        
        try {
            $user->notify(new SecurityAlertNotification($event, $alertModel));
            
            // Update rate limiting cache
            $this->updateRateLimit($domain, $event->alertType);
            
            Log::info('Security alert notification sent', [
                'domain_id' => $domain->id,
                'domain' => $domain->domain,
                'user_id' => $user->id,
                'alert_type' => $event->alertType,
                'alert_level' => $event->alertLevel
            ]);
            
        } catch (\Exception $e) {
            Log::error('Failed to send security alert notification', [
                'domain_id' => $domain->id,
                'domain' => $domain->domain,
                'user_id' => $user->id,
                'alert_type' => $event->alertType,
                'error' => $e->getMessage()
            ]);
        }
    }
    
    /**
     * Update domain security status based on alert.
     */
    protected function updateDomainSecurityStatus(SecurityAlert $event): void
    {
        $domain = $event->domain;
        
        // Update domain security status based on alert level
        switch ($event->alertLevel) {
            case 'error':
                if ($domain->security_status !== 'critical') {
                    $domain->update([
                        'security_status' => 'critical',
                        'last_security_check' => now()
                    ]);
                }
                break;
                
            case 'warning':
                if ($domain->security_status === 'good') {
                    $domain->update([
                        'security_status' => 'warning',
                        'last_security_check' => now()
                    ]);
                }
                break;
        }
        
        // Update trust seal status if security is compromised
        if (in_array($event->alertLevel, ['error']) && $event->alertType === 'malware') {
            $trustSeal = $domain->trustSeal;
            if ($trustSeal && $trustSeal->is_active) {
                $trustSeal->update(['is_active' => false]);
                
                Log::warning('Trust seal deactivated due to security alert', [
                    'domain_id' => $domain->id,
                    'seal_id' => $trustSeal->id,
                    'alert_type' => $event->alertType
                ]);
            }
        }
    }
    
    /**
     * Log the security alert.
     */
    protected function logSecurityAlert(SecurityAlert $event): void
    {
        Log::info('Security alert processed', [
            'domain_id' => $event->domain->id,
            'domain' => $event->domain->domain,
            'user_id' => $event->domain->user_id,
            'alert_type' => $event->alertType,
            'alert_level' => $event->alertLevel,
            'message' => $event->message,
            'priority' => $event->getPriority(),
            'alerted_at' => $event->alertedAt->toISOString()
        ]);
    }
    
    /**
     * Update security statistics and cache.
     */
    protected function updateSecurityStats(SecurityAlert $event): void
    {
        try {
            $domain = $event->domain;
            $user = $domain->user;
            
            // Update user's security alert count cache
            $cacheKey = "user_security_alerts_{$user->id}";
            $alertCount = Cache::get($cacheKey, 0) + 1;
            Cache::put($cacheKey, $alertCount, now()->addDay());
            
            // Update domain security alert count
            $domainCacheKey = "domain_security_alerts_{$domain->id}";
            $domainAlertCount = Cache::get($domainCacheKey, 0) + 1;
            Cache::put($domainCacheKey, $domainAlertCount, now()->addDay());
            
            Log::info('Security stats updated', [
                'user_id' => $user->id,
                'domain_id' => $domain->id,
                'user_alert_count' => $alertCount,
                'domain_alert_count' => $domainAlertCount
            ]);
            
        } catch (\Exception $e) {
            Log::warning('Failed to update security stats', [
                'domain_id' => $event->domain->id,
                'error' => $e->getMessage()
            ]);
        }
    }
    
    /**
     * Check if we should notify for this alert level.
     */
    protected function shouldNotifyForLevel(string $alertLevel, string $minLevel): bool
    {
        $levels = ['info' => 1, 'warning' => 2, 'error' => 3];
        
        return ($levels[$alertLevel] ?? 1) >= ($levels[$minLevel] ?? 2);
    }
    
    /**
     * Check if notifications are rate limited for this domain and alert type.
     */
    protected function isRateLimited($domain, string $alertType): bool
    {
        $cacheKey = "security_alert_rate_limit_{$domain->id}_{$alertType}";
        
        return Cache::has($cacheKey);
    }
    
    /**
     * Update rate limiting cache.
     */
    protected function updateRateLimit($domain, string $alertType): void
    {
        $cacheKey = "security_alert_rate_limit_{$domain->id}_{$alertType}";
        
        // Rate limit based on alert type
        $ttl = match($alertType) {
            'malware' => now()->addHours(6), // 6 hours for malware alerts
            'ssl' => now()->addHours(12),    // 12 hours for SSL alerts
            'dns' => now()->addHours(4),     // 4 hours for DNS alerts
            'safe_browsing' => now()->addHours(8), // 8 hours for safe browsing
            default => now()->addHours(2)    // 2 hours for other alerts
        };
        
        Cache::put($cacheKey, true, $ttl);
    }
    
    /**
     * Handle a job failure.
     */
    public function failed(SecurityAlert $event, \Throwable $exception): void
    {
        Log::error('Security alert listener failed', [
            'domain_id' => $event->domain->id,
            'domain' => $event->domain->domain,
            'alert_type' => $event->alertType,
            'alert_level' => $event->alertLevel,
            'error' => $exception->getMessage(),
            'trace' => $exception->getTraceAsString()
        ]);
    }
}