<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use App\Models\Domain;
use Symfony\Component\HttpFoundation\Response;

class VerifyDomainOwnership
{
    /**
     * Handle an incoming request.
     *
     * @param  \Closure(\Illuminate\Http\Request): (\Symfony\Component\HttpFoundation\Response)  $next
     */
    public function handle(Request $request, Closure $next): Response
    {
        $user = $request->user();
        
        if (!$user) {
            return response()->json([
                'error' => 'Unauthorized',
                'message' => 'Authentication required'
            ], 401);
        }
        
        // Get domain from route parameter or request
        $domainParam = $request->route('domain') ?? $request->get('domain');
        
        if (!$domainParam) {
            return response()->json([
                'error' => 'Bad Request',
                'message' => 'Domain parameter is required'
            ], 400);
        }
        
        // Handle both Domain model and domain string
        if ($domainParam instanceof Domain) {
            $domain = $domainParam;
        } else {
            $domain = Domain::where('domain', $domainParam)->first();
        }
        
        if (!$domain) {
            return response()->json([
                'error' => 'Not Found',
                'message' => 'Domain not found'
            ], 404);
        }
        
        // Verify ownership
        if ($domain->user_id !== $user->id) {
            return response()->json([
                'error' => 'Forbidden',
                'message' => 'You do not own this domain'
            ], 403);
        }
        
        // Add domain to request for use in controller
        $request->merge(['verified_domain' => $domain]);
        
        return $next($request);
    }
}