<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Http\Response;
use App\Models\TrustSeal;
use App\Models\SealAnalytics;
use App\Services\GeoLocationService;
use Jenssegers\Agent\Agent;
use Carbon\Carbon;

class TrustSealController extends Controller
{
    protected $geoService;
    
    public function __construct(GeoLocationService $geoService)
    {
        $this->geoService = $geoService;
    }

    /**
     * Display the trust seal (public endpoint)
     */
    public function show(Request $request, $sealCode)
    {
        $seal = TrustSeal::where('seal_code', $sealCode)
            ->where('is_active', true)
            ->with(['domain.securityChecks' => function($query) {
                $query->latest()->limit(5);
            }])
            ->first();
            
        if (!$seal || !$seal->domain->is_active) {
            return $this->renderErrorSeal('Seal not found or inactive');
        }
        
        // Track the view
        $this->trackView($request, $seal);
        
        // Get seal data
        $sealData = $this->prepareSealData($seal);
        
        // Return iframe content
        return response()->view('seals.display', compact('seal', 'sealData'))
            ->header('X-Frame-Options', 'ALLOWALL')
            ->header('Content-Security-Policy', "frame-ancestors *;");
    }
    
    /**
     * Get seal data as JSON (for AJAX requests)
     */
    public function api(Request $request, $sealCode)
    {
        $seal = TrustSeal::where('seal_code', $sealCode)
            ->where('is_active', true)
            ->with(['domain.securityChecks' => function($query) {
                $query->latest()->limit(5);
            }])
            ->first();
            
        if (!$seal || !$seal->domain->is_active) {
            return response()->json([
                'error' => 'Seal not found or inactive',
                'seal_type' => 'error'
            ], 404);
        }
        
        // Track the view
        $this->trackView($request, $seal);
        
        // Get seal data
        $sealData = $this->prepareSealData($seal);
        
        return response()->json($sealData);
    }
    
    /**
     * Generate embed code for domain owner
     */
    public function embed(TrustSeal $seal)
    {
        $this->authorize('view', $seal->domain);
        
        $embedCode = $seal->getEmbedCode();
        $directLink = $seal->getDirectLink();
        
        return view('seals.embed', compact('seal', 'embedCode', 'directLink'));
    }
    
    /**
     * Update seal customization
     */
    public function updateCustomization(Request $request, TrustSeal $seal)
    {
        $this->authorize('update', $seal->domain);
        
        $request->validate([
            'theme' => 'nullable|in:light,dark,blue,green,custom',
            'size' => 'nullable|in:small,medium,large',
            'position' => 'nullable|in:bottom-right,bottom-left,top-right,top-left',
            'show_score' => 'boolean',
            'show_domain' => 'boolean',
            'show_verified_date' => 'boolean',
            'custom_color' => 'nullable|string|max:7|regex:/^#[0-9a-fA-F]{6}$/',
            'custom_text' => 'nullable|string|max:100'
        ]);
        
        $customization = array_merge($seal->getDefaultCustomization(), $request->only([
            'theme', 'size', 'position', 'show_score', 'show_domain', 
            'show_verified_date', 'custom_color', 'custom_text'
        ]));
        
        $seal->update([
            'customization' => $customization
        ]);
        
        return redirect()->back()
            ->with('success', 'Seal customization updated successfully.');
    }
    
    /**
     * Toggle seal status
     */
    public function toggleStatus(TrustSeal $seal)
    {
        $this->authorize('update', $seal->domain);
        
        $seal->update([
            'is_active' => !$seal->is_active
        ]);
        
        $status = $seal->is_active ? 'activated' : 'deactivated';
        
        return redirect()->back()
            ->with('success', "Trust seal has been {$status} successfully.");
    }
    
    /**
     * Get seal statistics
     */
    public function statistics(TrustSeal $seal)
    {
        $this->authorize('view', $seal->domain);
        
        $stats = [
            'total_views' => $seal->display_count,
            'views_today' => $seal->sealAnalytics()->today()->count(),
            'views_this_week' => $seal->sealAnalytics()->thisWeek()->count(),
            'views_this_month' => $seal->sealAnalytics()->thisMonth()->count(),
            'top_countries' => $seal->sealAnalytics()->getTopCountries(5),
            'top_browsers' => $seal->sealAnalytics()->getTopBrowsers(5),
            'device_breakdown' => $seal->sealAnalytics()->getDeviceBreakdown(),
            'daily_views' => $this->getDailyViews($seal, 30)
        ];
        
        return response()->json($stats);
    }
    
    /**
     * Track seal view
     */
    private function trackView(Request $request, TrustSeal $seal)
    {
        $agent = new Agent();
        $agent->setUserAgent($request->userAgent());
        
        // Get location data
        $locationData = $this->geoService->getLocationData($request->ip());
        
        // Create analytics record
        SealAnalytics::create([
            'trust_seal_id' => $seal->id,
            'ip_address' => $request->ip(),
            'user_agent' => $request->userAgent(),
            'referrer' => $request->header('referer'),
            'country_code' => $locationData['country_code'] ?? null,
            'city' => $locationData['city'] ?? null,
            'browser' => $agent->browser(),
            'os' => $agent->platform(),
            'device_type' => $agent->deviceType(),
            'viewed_at' => now()
        ]);
        
        // Increment display count
        $seal->increment('display_count');
        
        // Update first/last displayed timestamps
        if (!$seal->first_displayed_at) {
            $seal->update(['first_displayed_at' => now()]);
        }
        $seal->update(['last_displayed_at' => now()]);
    }
    
    /**
     * Prepare seal data for display
     */
    private function prepareSealData(TrustSeal $seal)
    {
        $domain = $seal->domain;
        $latestSecurityCheck = $domain->securityChecks->first();
        
        // Determine seal type based on verification status
        $sealType = 'trust'; // Default
        if ($domain->isVerified() && $domain->hasValidSSL() && $domain->canDisplaySecuritySeal()) {
            $sealType = 'security';
        }
        
        return [
            'seal_code' => $seal->seal_code,
            'seal_type' => $sealType,
            'domain' => $domain->getCleanDomain(),
            'verified_at' => $domain->verified_at ? $domain->verified_at->format('M j, Y') : null,
            'security_score' => $domain->security_score ?? 0,
            'security_score_color' => $domain->getSecurityScoreColor(),
            'ssl_valid' => $domain->hasValidSSL(),
            'ssl_expires' => $domain->ssl_info['expires_at'] ?? null,
            'last_check' => $latestSecurityCheck ? $latestSecurityCheck->checked_at->diffForHumans() : null,
            'customization' => $seal->customization ?? $seal->getDefaultCustomization(),
            'display_count' => $seal->display_count,
            'timestamp' => now()->timestamp
        ];
    }
    
    /**
     * Render error seal
     */
    private function renderErrorSeal(string $message)
    {
        $sealData = [
            'seal_type' => 'error',
            'message' => $message,
            'customization' => [
                'theme' => 'light',
                'size' => 'medium'
            ]
        ];
        
        return response()->view('seals.error', compact('sealData'))
            ->header('X-Frame-Options', 'ALLOWALL')
            ->header('Content-Security-Policy', "frame-ancestors *;");
    }
    
    /**
     * Get daily views for the last N days
     */
    private function getDailyViews(TrustSeal $seal, int $days = 30)
    {
        $dailyViews = [];
        
        for ($i = $days - 1; $i >= 0; $i--) {
            $date = now()->subDays($i);
            $views = $seal->sealAnalytics()
                ->whereDate('viewed_at', $date->toDateString())
                ->count();
                
            $dailyViews[] = [
                'date' => $date->format('Y-m-d'),
                'views' => $views
            ];
        }
        
        return $dailyViews;
    }
}