<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\View\View;
use App\Models\Domain;
use App\Models\TrustSeal;
use App\Models\User;
use App\Models\SealAnalytics;
use App\Services\JsonAuthService;

class HomeController extends Controller
{
    protected $jsonAuth;
    
    public function __construct(JsonAuthService $jsonAuth)
    {
        $this->jsonAuth = $jsonAuth;
    }
    
    /**
     * Show the application homepage.
     */
    public function index(): View
    {
        // Check if user is authenticated
        if ($this->jsonAuth->check()) {
            return $this->dashboard();
        }
        
        // Get some statistics for the homepage (using demo data)
        $stats = [
            'verified_domains' => 15,
            'active_seals' => 25,
            'total_users' => 150,
            'seal_views' => 5420
        ];

        // Get recent testimonials or featured domains (mock data for now)
        $testimonials = [
            [
                'name' => 'John Smith',
                'company' => 'TechCorp Inc.',
                'message' => 'Trust Seal Portal helped us increase customer confidence by 40%. The security monitoring is excellent!',
                'rating' => 5
            ],
            [
                'name' => 'Sarah Johnson',
                'company' => 'E-Commerce Plus',
                'message' => 'Easy to implement and great analytics. Our conversion rate improved significantly.',
                'rating' => 5
            ],
            [
                'name' => 'Mike Chen',
                'company' => 'Digital Solutions',
                'message' => 'Professional service with excellent customer support. Highly recommended!',
                'rating' => 5
            ]
        ];

        return view('home.index', compact('stats', 'testimonials'));
    }

    /**
     * Show the features page.
     */
    public function features(): View
    {
        $features = [
            [
                'title' => 'Domain Verification',
                'description' => 'Verify domain ownership through multiple methods including DNS, file upload, and meta tags.',
                'icon' => 'shield-check',
                'benefits' => [
                    'Multiple verification methods',
                    'Automatic SSL certificate monitoring',
                    'Real-time status updates',
                    'Bulk domain management'
                ]
            ],
            [
                'title' => 'Security Monitoring',
                'description' => 'Continuous security monitoring with SSL certificate tracking and vulnerability scanning.',
                'icon' => 'security',
                'benefits' => [
                    'SSL certificate monitoring',
                    'DNS health checks',
                    'Malware detection',
                    'Security score tracking'
                ]
            ]
        ];

        return view('home.features', compact('features'));
    }

    /**
     * Show the pricing page.
     */
    public function pricing(): View
    {
        $plans = [
            [
                'name' => 'Basic',
                'price' => 0,
                'billing' => 'Free Forever',
                'description' => 'Perfect for small websites and personal projects',
                'features' => [
                    '1 Domain',
                    'Basic trust seal',
                    'SSL monitoring',
                    '1,000 seal views/month',
                    'Basic analytics',
                    'Email support'
                ],
                'popular' => false
            ],
            [
                'name' => 'Professional',
                'price' => 29,
                'billing' => 'per month',
                'description' => 'Ideal for growing businesses and e-commerce sites',
                'features' => [
                    '10 Domains',
                    'All seal designs',
                    'Advanced security monitoring',
                    '50,000 seal views/month',
                    'Advanced analytics',
                    'API access'
                ],
                'popular' => true
            ]
        ];

        return view('home.pricing', compact('plans'));
    }

    /**
     * Show the about page.
     */
    public function about(): View
    {
        $team = [
            [
                'name' => 'Alex Johnson',
                'role' => 'CEO & Founder',
                'bio' => 'Former security engineer with 10+ years experience in web security and trust systems.',
                'image' => 'team/alex.jpg'
            ],
            [
                'name' => 'Sarah Chen',
                'role' => 'CTO',
                'bio' => 'Full-stack developer and security expert, passionate about building secure web applications.',
                'image' => 'team/sarah.jpg'
            ],
            [
                'name' => 'Mike Rodriguez',
                'role' => 'Head of Customer Success',
                'bio' => 'Dedicated to helping customers maximize their trust and security initiatives.',
                'image' => 'team/mike.jpg'
            ]
        ];

        return view('home.about', compact('team'));
    }

    /**
     * Show the contact page.
     */
    public function contact(): View
    {
        return view('home.contact');
    }

    /**
     * Handle contact form submission.
     */
    public function contactSubmit(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|max:255',
            'subject' => 'required|string|max:255',
            'message' => 'required|string|max:2000',
            'company' => 'nullable|string|max:255'
        ]);

        // In a real application, you would send emails and save to database
        \Log::info('Contact form submission', [
            'name' => $request->name,
            'email' => $request->email,
            'subject' => $request->subject
        ]);

        return redirect()->route('contact')
            ->with('success', 'Thank you for your message! We\'ll get back to you within 24 hours.');
    }

    /**
     * Show the privacy policy page.
     */
    public function privacy(): View
    {
        return view('home.privacy');
    }

    /**
     * Show the terms of service page.
     */
    public function terms(): View
    {
        return view('home.terms');
    }

    /**
     * Show the help/documentation page.
     */
    public function help(): View
    {
        $helpSections = [
            [
                'title' => 'Getting Started',
                'articles' => [
                    ['title' => 'How to verify your domain', 'slug' => 'verify-domain'],
                    ['title' => 'Setting up your first trust seal', 'slug' => 'setup-trust-seal']
                ]
            ],
            [
                'title' => 'Domain Management',
                'articles' => [
                    ['title' => 'Adding multiple domains', 'slug' => 'multiple-domains'],
                    ['title' => 'Domain verification methods', 'slug' => 'verification-methods']
                ]
            ]
        ];

        return view('home.help', compact('helpSections'));
    }
    
    /**
     * Show the user dashboard
     */
    public function dashboard()
    {
        $user = $this->jsonAuth->user();
        $stats = $this->jsonAuth->getUserStats($user['id']);
        $domains = $this->jsonAuth->getDomains($user['id']);
        
        // Add is_verified field based on verification_status
        $domains = array_map(function($domain) {
            $domain['is_verified'] = ($domain['verification_status'] === 'verified');
            return $domain;
        }, $domains);
        
        return view('dashboard', compact('user', 'stats', 'domains'));
    }
    
    /**
     * Show login form
     */
    public function login()
    {
        if ($this->jsonAuth->check()) {
            return redirect()->route('home');
        }
        
        return view('auth.login');
    }
    
    /**
     * Handle login attempt
     */
    public function loginPost(Request $request)
    {
        $request->validate([
            'email' => 'required|email',
            'password' => 'required'
        ]);
        
        if ($this->jsonAuth->attempt($request->email, $request->password)) {
            return redirect()->route('home')->with('success', 'Successfully logged in!');
        }
        
        return back()->withErrors(['email' => 'Invalid credentials'])->withInput();
    }
    
    /**
     * Handle logout
     */
    public function logout()
    {
        $this->jsonAuth->logout();
        return redirect()->route('home')->with('success', 'Successfully logged out!');
    }
    
    /**
     * Show admin panel
     */
    public function admin()
    {
        if (!$this->jsonAuth->isAdmin()) {
            return redirect()->route('home')->with('error', 'Access denied. Admin privileges required.');
        }
        
        // Get private method data for admin
        $allUsers = $this->getAllUsersForAdmin();
        $allDomains = $this->jsonAuth->getDomains();
        $allSeals = $this->jsonAuth->getTrustSeals();
        
        // Add is_verified field based on verification_status
        $allDomains = array_map(function($domain) {
            $domain['is_verified'] = ($domain['verification_status'] === 'verified');
            return $domain;
        }, $allDomains);
        
        $adminStats = [
            'total_users' => count($allUsers),
            'total_domains' => count($allDomains),
            'total_seals' => count($allSeals),
            'verified_domains' => count(array_filter($allDomains, function($d) { return $d['verification_status'] === 'verified'; }))
        ];
        
        return view('admin', compact('adminStats', 'allUsers', 'allDomains'));
    }
    
    /**
     * Private method to get all users for admin
     */
    private function getAllUsersForAdmin()
    {
        $filePath = storage_path('app/demo-data/users.json');
        
        if (!file_exists($filePath)) {
            return [];
        }
        
        $content = file_get_contents($filePath);
        return json_decode($content, true) ?: [];
    }
    
    /**
     * Show domain creation form
     */
    public function createDomain()
    {
        if (!$this->jsonAuth->isAuthenticated()) {
            return redirect()->route('login')->with('error', 'Please login to add domains.');
        }
        
        return view('domains.create');
    }
    
    /**
     * Store new domain
     */
    public function storeDomain(Request $request)
    {
        if (!$this->jsonAuth->isAuthenticated()) {
            return redirect()->route('login')->with('error', 'Please login to add domains.');
        }
        
        $request->validate([
            'domain' => 'required|string|max:255|regex:/^[a-zA-Z0-9.-]+\.[a-zA-Z]{2,}$/'
        ]);
        
        $user = $this->jsonAuth->user();
        $cleanDomain = preg_replace('/^https?:\/\//', '', $request->domain);
        $cleanDomain = rtrim($cleanDomain, '/');
        
        $result = $this->jsonAuth->addDomain($user['id'], $cleanDomain);
        
        if (isset($result['error'])) {
            return redirect()->back()->with('error', $result['error'])->withInput();
        }
        
        return redirect()->route('dashboard')->with('success', 'Domain added successfully!');
    }
}
