<?php

namespace App\Http\Controllers\Api;

use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use App\Models\Domain;
use App\Models\TrustSeal;
use App\Services\AnalyticsService;
use App\Services\SecurityMonitoringService;
use Illuminate\Http\JsonResponse;
use Illuminate\Validation\ValidationException;

class ApiController extends Controller
{
    protected $analyticsService;
    protected $securityService;

    public function __construct(AnalyticsService $analyticsService, SecurityMonitoringService $securityService)
    {
        $this->middleware('auth:sanctum');
        $this->analyticsService = $analyticsService;
        $this->securityService = $securityService;
    }

    /**
     * Get user's domains.
     */
    public function domains(Request $request): JsonResponse
    {
        $user = Auth::user();
        
        $domains = $user->domains()
            ->with(['trustSeal', 'latestSecurityCheck'])
            ->when($request->verified, function ($query) {
                return $query->where('status', 'verified');
            })
            ->when($request->active, function ($query) {
                return $query->where('is_active', true);
            })
            ->paginate($request->per_page ?? 15);

        return response()->json([
            'status' => 'success',
            'data' => $domains->items(),
            'pagination' => [
                'current_page' => $domains->currentPage(),
                'last_page' => $domains->lastPage(),
                'per_page' => $domains->perPage(),
                'total' => $domains->total()
            ]
        ]);
    }

    /**
     * Get domain details.
     */
    public function domain(Request $request, $id): JsonResponse
    {
        $user = Auth::user();
        
        $domain = $user->domains()
            ->with(['trustSeal', 'securityChecks', 'verifications'])
            ->findOrFail($id);

        return response()->json([
            'status' => 'success',
            'data' => [
                'id' => $domain->id,
                'domain' => $domain->domain,
                'status' => $domain->status,
                'verified_at' => $domain->verified_at,
                'security_score' => $domain->security_score,
                'ssl_info' => $domain->ssl_info,
                'is_active' => $domain->is_active,
                'trust_seal' => $domain->trustSeal ? [
                    'id' => $domain->trustSeal->id,
                    'seal_type' => $domain->trustSeal->seal_type,
                    'is_active' => $domain->trustSeal->is_active,
                    'display_count' => $domain->trustSeal->display_count,
                    'embed_code' => $domain->trustSeal->getEmbedCode(),
                    'direct_link' => $domain->trustSeal->getDirectLink()
                ] : null,
                'latest_security_check' => $domain->latestSecurityCheck,
                'created_at' => $domain->created_at,
                'updated_at' => $domain->updated_at
            ]
        ]);
    }

    /**
     * Add a new domain.
     */
    public function addDomain(Request $request): JsonResponse
    {
        $user = Auth::user();
        
        // Check domain limit
        if (!$user->canAddDomain()) {
            return response()->json([
                'status' => 'error',
                'message' => 'Domain limit reached. Please upgrade your plan to add more domains.',
                'errors' => ['domain' => ['Domain limit exceeded']]
            ], 422);
        }

        try {
            $request->validate([
                'domain' => [
                    'required',
                    'string',
                    'max:255',
                    'regex:/^(?:[a-zA-Z0-9](?:[a-zA-Z0-9-]{0,61}[a-zA-Z0-9])?\.)+(xn--[a-zA-Z0-9]+|[a-zA-Z]{2,})$/',
                    function ($attribute, $value, $fail) use ($user) {
                        $cleanDomain = preg_replace('/^https?:\/\//', '', strtolower(trim($value)));
                        $cleanDomain = rtrim($cleanDomain, '/');
                        
                        if ($user->domains()->where('domain', $cleanDomain)->exists()) {
                            $fail('This domain is already added to your account.');
                        }
                    }
                ]
            ]);
        } catch (ValidationException $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Validation failed',
                'errors' => $e->errors()
            ], 422);
        }

        $cleanDomain = preg_replace('/^https?:\/\//', '', strtolower(trim($request->domain)));
        $cleanDomain = rtrim($cleanDomain, '/');

        $domain = Domain::create([
            'user_id' => $user->id,
            'domain' => $cleanDomain,
            'status' => 'pending',
            'verification_token' => bin2hex(random_bytes(32)),
            'is_active' => true
        ]);

        return response()->json([
            'status' => 'success',
            'message' => 'Domain added successfully',
            'data' => [
                'id' => $domain->id,
                'domain' => $domain->domain,
                'status' => $domain->status,
                'verification_token' => $domain->verification_token,
                'created_at' => $domain->created_at
            ]
        ], 201);
    }

    /**
     * Update domain status.
     */
    public function updateDomain(Request $request, $id): JsonResponse
    {
        $user = Auth::user();
        $domain = $user->domains()->findOrFail($id);

        $request->validate([
            'is_active' => 'sometimes|boolean'
        ]);

        if ($request->has('is_active')) {
            $domain->update(['is_active' => $request->is_active]);
            
            // Also update trust seal status if it exists
            if ($domain->trustSeal) {
                $domain->trustSeal->update(['is_active' => $request->is_active]);
            }
        }

        return response()->json([
            'status' => 'success',
            'message' => 'Domain updated successfully',
            'data' => [
                'id' => $domain->id,
                'domain' => $domain->domain,
                'is_active' => $domain->is_active
            ]
        ]);
    }

    /**
     * Get domain analytics.
     */
    public function domainAnalytics(Request $request, $id): JsonResponse
    {
        $user = Auth::user();
        $domain = $user->domains()->findOrFail($id);

        $request->validate([
            'period' => 'sometimes|in:today,week,month,year',
            'timezone' => 'sometimes|string'
        ]);

        $period = $request->period ?? 'month';
        $analytics = $this->analyticsService->getDomainAnalytics($domain->id, $period);

        return response()->json([
            'status' => 'success',
            'data' => [
                'domain' => $domain->domain,
                'period' => $period,
                'analytics' => $analytics
            ]
        ]);
    }

    /**
     * Get user analytics summary.
     */
    public function userAnalytics(Request $request): JsonResponse
    {
        $user = Auth::user();

        $request->validate([
            'period' => 'sometimes|in:today,week,month,year'
        ]);

        $period = $request->period ?? 'month';
        $analytics = $this->analyticsService->getUserAnalytics($user->id, $period);

        return response()->json([
            'status' => 'success',
            'data' => [
                'period' => $period,
                'analytics' => $analytics
            ]
        ]);
    }

    /**
     * Get trust seal data.
     */
    public function trustSeal(Request $request, $id): JsonResponse
    {
        $user = Auth::user();
        $domain = $user->domains()->findOrFail($id);
        
        if (!$domain->trustSeal) {
            return response()->json([
                'status' => 'error',
                'message' => 'Trust seal not found for this domain'
            ], 404);
        }

        $trustSeal = $domain->trustSeal;

        return response()->json([
            'status' => 'success',
            'data' => [
                'id' => $trustSeal->id,
                'domain' => $domain->domain,
                'seal_type' => $trustSeal->seal_type,
                'seal_code' => $trustSeal->seal_code,
                'is_active' => $trustSeal->is_active,
                'display_count' => $trustSeal->display_count,
                'customization' => $trustSeal->customization,
                'embed_code' => $trustSeal->getEmbedCode(),
                'direct_link' => $trustSeal->getDirectLink(),
                'first_displayed_at' => $trustSeal->first_displayed_at,
                'last_displayed_at' => $trustSeal->last_displayed_at,
                'created_at' => $trustSeal->created_at
            ]
        ]);
    }

    /**
     * Update trust seal configuration.
     */
    public function updateTrustSeal(Request $request, $id): JsonResponse
    {
        $user = Auth::user();
        $domain = $user->domains()->findOrFail($id);
        
        if (!$domain->trustSeal) {
            return response()->json([
                'status' => 'error',
                'message' => 'Trust seal not found for this domain'
            ], 404);
        }

        $request->validate([
            'seal_type' => 'sometimes|in:basic,professional,enterprise',
            'is_active' => 'sometimes|boolean',
            'customization' => 'sometimes|array',
            'customization.color' => 'sometimes|string|max:7',
            'customization.size' => 'sometimes|in:small,medium,large',
            'customization.position' => 'sometimes|in:bottom-right,bottom-left,top-right,top-left',
            'customization.animation' => 'sometimes|boolean'
        ]);

        $trustSeal = $domain->trustSeal;
        $updateData = [];

        if ($request->has('seal_type')) {
            $updateData['seal_type'] = $request->seal_type;
        }

        if ($request->has('is_active')) {
            $updateData['is_active'] = $request->is_active;
        }

        if ($request->has('customization')) {
            $currentCustomization = $trustSeal->customization ?? [];
            $newCustomization = array_merge($currentCustomization, $request->customization);
            $updateData['customization'] = $newCustomization;
        }

        $trustSeal->update($updateData);

        return response()->json([
            'status' => 'success',
            'message' => 'Trust seal updated successfully',
            'data' => [
                'id' => $trustSeal->id,
                'seal_type' => $trustSeal->seal_type,
                'is_active' => $trustSeal->is_active,
                'customization' => $trustSeal->customization,
                'embed_code' => $trustSeal->getEmbedCode()
            ]
        ]);
    }

    /**
     * Run security check for domain.
     */
    public function runSecurityCheck(Request $request, $id): JsonResponse
    {
        $user = Auth::user();
        $domain = $user->domains()->findOrFail($id);

        // Check if user has permission to run security checks
        if (!$user->isPremium() && $domain->securityChecks()->where('created_at', '>=', now()->subDay())->count() >= 1) {
            return response()->json([
                'status' => 'error',
                'message' => 'Free users can only run one security check per day per domain. Please upgrade to premium for unlimited checks.'
            ], 429);
        }

        try {
            $results = $this->securityService->runSecurityCheck($domain);
            
            return response()->json([
                'status' => 'success',
                'message' => 'Security check completed',
                'data' => [
                    'domain' => $domain->domain,
                    'overall_score' => $results['overall_score'],
                    'checks' => $results['checks'],
                    'checked_at' => now()->toISOString()
                ]
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Security check failed: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get security alerts.
     */
    public function securityAlerts(Request $request): JsonResponse
    {
        $user = Auth::user();
        $alerts = $this->securityService->getSecurityAlerts($user->id);

        return response()->json([
            'status' => 'success',
            'data' => [
                'alerts' => $alerts,
                'count' => count($alerts)
            ]
        ]);
    }

    /**
     * Get real-time statistics.
     */
    public function realTimeStats(Request $request): JsonResponse
    {
        $user = Auth::user();
        $stats = $this->analyticsService->getRealTimeStats($user->id);

        return response()->json([
            'status' => 'success',
            'data' => $stats
        ]);
    }

    /**
     * Export analytics data.
     */
    public function exportAnalytics(Request $request): JsonResponse
    {
        $user = Auth::user();

        $request->validate([
            'period' => 'sometimes|in:week,month,quarter,year',
            'format' => 'sometimes|in:json,csv',
            'domain_id' => 'sometimes|exists:domains,id'
        ]);

        $period = $request->period ?? 'month';
        $format = $request->format ?? 'json';
        
        try {
            if ($request->domain_id) {
                $domain = $user->domains()->findOrFail($request->domain_id);
                $data = $this->analyticsService->getDomainAnalytics($domain->id, $period);
                $filename = "domain_{$domain->domain}_{$period}_analytics";
            } else {
                $data = $this->analyticsService->getUserAnalytics($user->id, $period);
                $filename = "user_{$user->id}_{$period}_analytics";
            }

            if ($format === 'csv') {
                $csvData = $this->analyticsService->exportAnalytics($user->id, $period, $format);
                
                return response()->json([
                    'status' => 'success',
                    'data' => [
                        'filename' => $filename . '.csv',
                        'content' => base64_encode($csvData),
                        'mime_type' => 'text/csv'
                    ]
                ]);
            } else {
                return response()->json([
                    'status' => 'success',
                    'data' => [
                        'filename' => $filename . '.json',
                        'content' => $data,
                        'exported_at' => now()->toISOString()
                    ]
                ]);
            }
        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Export failed: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get API usage statistics.
     */
    public function apiUsage(Request $request): JsonResponse
    {
        $user = Auth::user();
        
        // In a real application, you would track API usage
        $usage = [
            'requests_today' => 0,
            'requests_this_month' => 0,
            'rate_limit' => $user->isPremium() ? 10000 : 1000,
            'rate_limit_remaining' => $user->isPremium() ? 9950 : 950,
            'rate_limit_reset' => now()->addHour()->timestamp
        ];

        return response()->json([
            'status' => 'success',
            'data' => $usage
        ]);
    }

    /**
     * Handle API errors consistently.
     */
    protected function errorResponse(string $message, int $code = 400, array $errors = []): JsonResponse
    {
        $response = [
            'status' => 'error',
            'message' => $message
        ];

        if (!empty($errors)) {
            $response['errors'] = $errors;
        }

        return response()->json($response, $code);
    }

    /**
     * Handle successful API responses consistently.
     */
    protected function successResponse($data = null, string $message = 'Success', int $code = 200): JsonResponse
    {
        $response = [
            'status' => 'success',
            'message' => $message
        ];

        if ($data !== null) {
            $response['data'] = $data;
        }

        return response()->json($response, $code);
    }
}